'use strict';

/* --------------------------------------------------------------
 datepicker.js 2025-04-22
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2025 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Datepicker Widget
 *
 * Creates a customizable date(range)picker.
 *
 * jQueryUI Datepicker API: {@link http://api.jqueryui.com/datepicker}
 *
 * You can add the `data-datepicker-gx-container` attribute and it will style the datepicker with
 * the new CSS styles located at the gx-admin.css file. This might be useful when the .gx-container
 * class is not set directly on the <body> tag but in an inner div element of the page. The datepicker
 * will create a new div element which might be outside the .gx-container and therefore will not have
 * its style. This widget is already styled in Honeygrid.
 *
 * ### Example
 *
 * When the page loads, an input field as a date picker will be added.
 *
 * ```html
 * <input type="text" data-jse-widget="datepicker" data-datepicker-show-On="focus"
 *      data-datepicker-gx-container placeholder="##.##.####" />
 * ```
 *
 * For custom date format, use the 'data-datepicker-format' attribute.
 *
 * @module JSE/Widgets/datepicker
 * @requires jQueryUI-Library
 */
jse.widgets.module('datepicker', [jse.source + '/vendor/jquery-ui-dist/jquery-ui.min.css', jse.source + '/vendor/jquery-ui-dist/jquery-ui.min.js'],

/** @lends module:Widgets/datepicker */

function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLE DEFINITION
    // ------------------------------------------------------------------------

    var
    /**
     * Widget Reference
     *
     * @type {object}
     */
    $this = $(this),


    /**
     * Default Options for Widget
     *
     * @type {object}
     */
    defaults = {},


    /**
     * Final Widget Options
     *
     * @type {object}
     */
    options = $.extend(true, {}, defaults, data),


    /**
     * Module Object
     *
     * @type {object}
     */
    module = {};

    // ------------------------------------------------------------------------
    // FUNCTIONALITY
    // ------------------------------------------------------------------------

    /**
     * Update Timestamp Field
     *
     * Function that updates the timestamp field belonging to this datepicker. If no
     * one exists, it gets generated.
     *
     * @param {object} inst jQuery datepicker instance object.
     */
    var _updateTimestampField = function _updateTimestampField(inst) {
        var name = $this.attr('name'),
            $ts = $this.siblings('[name="ts_' + name + '"]'),
            value = new Date([inst.selectedYear, inst.selectedMonth + 1, inst.selectedDay].join(', ')).valueOf();
        if (!$ts.length) {
            $this.after('<input type="hidden" name="ts_' + name + '" value="' + value + '"/>');
        } else {
            $ts.val(value);
        }
    };

    /**
     * Get Configuration
     *
     * Function to create the datepicker configuration object.
     *
     * @returns {object} JSON-configuration object.
     */
    var _getConfiguration = function _getConfiguration() {
        // Set default min / max values.
        options.max = options.max ? new Date(options.max) : null;
        options.min = options.min ? new Date(options.min) : null;

        // Base Configuration
        var configuration = {
            constrainInput: true,
            showOn: 'focus',
            showWeek: true,
            changeMonth: true,
            changeYear: true,
            minDate: options.min,
            maxDate: options.max,
            dayNamesMin: [jse.core.lang.translate('_SUNDAY_SHORT', 'general'), jse.core.lang.translate('_MONDAY_SHORT', 'general'), jse.core.lang.translate('_TUESDAY_SHORT', 'general'), jse.core.lang.translate('_WEDNESDAY_SHORT', 'general'), jse.core.lang.translate('_THURSDAY_SHORT', 'general'), jse.core.lang.translate('_FRIDAY_SHORT', 'general'), jse.core.lang.translate('_SATURDAY_SHORT', 'general')],
            dayNames: [jse.core.lang.translate('_SUNDAY', 'general'), jse.core.lang.translate('_MONDAY', 'general'), jse.core.lang.translate('_TUESDAY', 'general'), jse.core.lang.translate('_WEDNESDAY', 'general'), jse.core.lang.translate('_THURSDAY', 'general'), jse.core.lang.translate('_FRIDAY', 'general'), jse.core.lang.translate('_SATURDAY', 'general')],
            monthNamesShort: [jse.core.lang.translate('_JANUARY_SHORT', 'general'), jse.core.lang.translate('_FEBRUARY_SHORT', 'general'), jse.core.lang.translate('_MARCH_SHORT', 'general'), jse.core.lang.translate('_APRIL_SHORT', 'general'), jse.core.lang.translate('_MAY_SHORT', 'general'), jse.core.lang.translate('_JUNE_SHORT', 'general'), jse.core.lang.translate('_JULY_SHORT', 'general'), jse.core.lang.translate('_AUGUST_SHORT', 'general'), jse.core.lang.translate('_SEPTEMBER_SHORT', 'general'), jse.core.lang.translate('_OCTOBER_SHORT', 'general'), jse.core.lang.translate('_NOVEMBER_SHORT', 'general'), jse.core.lang.translate('_DECEMBER_SHORT', 'general')],
            monthNames: [jse.core.lang.translate('_JANUARY', 'general'), jse.core.lang.translate('_FEBRUARY', 'general'), jse.core.lang.translate('_MARCH', 'general'), jse.core.lang.translate('_APRIL', 'general'), jse.core.lang.translate('_MAY', 'general'), jse.core.lang.translate('_JUNE', 'general'), jse.core.lang.translate('_JULY', 'general'), jse.core.lang.translate('_AUGUST', 'general'), jse.core.lang.translate('_SEPTEMBER', 'general'), jse.core.lang.translate('_OCTOBER', 'general'), jse.core.lang.translate('_NOVEMBER', 'general'), jse.core.lang.translate('_DECEMBER', 'general')],
            prevText: jse.core.lang.translate('_PREV', 'general'),
            nextText: jse.core.lang.translate('_NEXT', 'general'),
            onSelect: function onSelect(date, inst) {
                _updateTimestampField(inst);
            }
        };

        // Set "showOn" options.
        if (options.showOn) {
            configuration.showOn = options.showOn;
        }

        // Sets the alternative field with another date format (for backend).
        if (options.alt) {
            configuration.altField = options.alt;
            configuration.altFormat = options.altFormat ? options.altFormat : '@';
        }

        // Trigger an event onSelect to inform dependencies and set the min / max value at the
        // current value of the dependency.
        if (options.depends && options.type) {
            var $depends = $(options.depends),
                value = $depends.val(),
                type = options.type === 'max' ? 'min' : 'max';

            // Add callback to the onSelect-Event.
            configuration.onSelect = function (date, inst) {
                _updateTimestampField(inst);
                var payload = {
                    'type': options.type,
                    'date': [inst.selectedYear, inst.selectedMonth + 1, inst.selectedDay].join(', ')
                };
                $depends.trigger('datepicker.selected', [payload]);
            };

            // Get and set the current value of the dependency.
            if (value) {
                var date = $.datepicker.parseDate($.datepicker._defaults.dateFormat, value);
                configuration[type + 'Date'] = date;
            }
        }

        // Override date format with data attribute value
        configuration.dateFormat = data.format;

        // Merge the data array with the datepicker array for enabling the original widget API options.
        configuration = $.extend(true, {}, configuration, data);

        return configuration;
    };

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    /**
     * Initialize method of the widget, called by the engine.
     */
    module.init = function (done) {
        // Enable the datepicker widget.
        var configuration = _getConfiguration();
        $this.datepicker(configuration);

        // Get the gx-container style (newer style).
        if (typeof options.gxContainer !== 'undefined') {
            $(document).find('.ui-datepicker').not('.gx-container').addClass('gx-container');
        }

        // Add event listener for other datepickers to set the min / maxDate (for daterange).
        $this.on('datepicker.selected', function (e, d) {
            $this.datepicker('option', d.type + 'Date', new Date(d.date));
        });

        done();
    };

    // Return data to module engine.
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
